(function() {
    let particles = [];
    let animating = false;

    const defaults = {
        amount: 150,
        colors: ["#ff3e3e", "#ffb32e", "#fff32e", "#2eff6e", "#2ecbff", "#b72eff"],
        size: { min: 4, max: 10 },
        gravity: 0.25,
        drift: { min: -0.4, max: 0.4 },
        speed: { min: -6, max: -12 },
        duration: 3000
    };

    let customSettings = { ...defaults };

    const canvas = document.createElement("canvas");
    canvas.style.position = "fixed";
    canvas.style.left = "0";
    canvas.style.top = "0";
    canvas.style.pointerEvents = "none";
    canvas.style.width = "100vw";
    canvas.style.height = "100vh";
    canvas.width = innerWidth;
    canvas.height = innerHeight;
    document.body.appendChild(canvas);

    const ctx = canvas.getContext("2d");

    function rand(min, max) {
        return Math.random() * (max - min) + min;
    }

    function createParticle(x, y, settings) {
        return {
            x,
            y,
            size: rand(settings.size.min, settings.size.max),
            color: settings.colors[Math.floor(Math.random() * settings.colors.length)],
            vx: rand(settings.drift.min, settings.drift.max),
            vy: rand(settings.speed.min, settings.speed.max),
            gravity: settings.gravity,
            life: settings.duration,
            maxLife: settings.duration
        };
    }

    function spawn(x, y, settings) {
        for (let i = 0; i < settings.amount; i++) {
            particles.push(createParticle(x, y, settings));
        }
        if (!animating) play();
    }

    function update() {
        ctx.clearRect(0, 0, canvas.width, canvas.height);

        particles = particles.filter(p => p.life > 0);

        for (let p of particles) {
            p.vy += p.gravity;
            p.x += p.vx;
            p.y += p.vy;
            p.life -= 16;

            const alpha = p.life / p.maxLife;
            ctx.globalAlpha = alpha;

            ctx.fillStyle = p.color;
            ctx.fillRect(p.x, p.y, p.size, p.size);
        }

        ctx.globalAlpha = 1;

        if (animating) {
            requestAnimationFrame(update);
        }
    }

    function play() {
        if (!animating) {
            animating = true;
            update();
        }
    }

    function pause() {
        animating = false;
    }

    // API
    window.Burst = {

        burst(amount = customSettings.amount) {
            spawn(innerWidth / 2, innerHeight / 2, { ...customSettings, amount });
        },

        burstAt(x, y, amount = customSettings.amount) {
            spawn(x, y, { ...customSettings, amount });
        },

        play,
        pause,

        custom(options = {}) {
            customSettings = {
                ...customSettings,
                ...options,
                size: { ...customSettings.size, ...(options.size || {}) },
                drift: { ...customSettings.drift, ...(options.drift || {}) },
                speed: { ...customSettings.speed, ...(options.speed || {}) }
            };
        },

        // Cambia directamente el tamaño (min y max)
        size(minSize, maxSize) {
            customSettings.size.min = minSize;
            customSettings.size.max = maxSize;
        }
    };

    addEventListener("resize", () => {
        canvas.width = innerWidth;
        canvas.height = innerHeight;
    });

})();
